// stdafx.h : include file for standard system include files,
// or project specific include files that are used frequently, but
// are changed infrequently
//

#pragma once

#include "targetver.h"

#define WIN32_LEAN_AND_MEAN             // Exclude rarely-used stuff from Windows headers
// Windows Header Files:
#include <windows.h>

// User Includes
#include <iostream>
#include <iomanip>
#include <cmath>
#include <vector>
#include <string>


// =============================================================================================
// SpatialSearch Original Header Stuff =========================================================
// =============================================================================================


// spatial.h (Joey Flamand)


// CONSTANTS
const int NUM_POINTS = 5000000;

const int QUERY_NO = 1;

const int MAX_PER_QUAD = 30; // number of returned points in each quadrant near the search point. 

// generated point constellation range 
const int RANGE_MAX = 1000;
const int RANGE_MIN = -1000;

// for a generated target point, define the range its ddistance can span
const int R_MIN = 1;
const int R_MAX = 3;

// QTree Limits
const int QTREE_MAX_LEVELS = 10; // only allow this many subdivisions deep ( (7)  4^6 = 4096 leaves )
const int QTREE_MAX_POINTS = 100; // split each subdivision until the number of points in this leaf node is below this 


// STRUCTURES
struct Point
{
  double x = 0;
  double y = 0;
};

struct PointDist
{
  double x = 0;
  double y = 0;
  double dist = 0;
};


// DECLARATIONS (Global)

double GetRandDouble(int low, int hi);

bool GenerateRandomPoints(Point *pPoints, unsigned int size);
void GenerateTargetPointEnvelope(PointDist &pPoint);

void PrintOnePoint(Point *onePoint);
void PrintOnePoint(PointDist *onePoint);
void PrintPointArray(Point *pPoints, unsigned int size);
void PrintPointDistArray(PointDist *pPoints, unsigned int size);

void FindConstellationExtents(std::vector<Point>* cloud,
  double &maxX, double &maxY, double &minX, double &minY);
void FindConstellationExtents(Point* pPoints, unsigned int size,
  double &maxX, double &maxY, double &minX, double &minY);

void ChopConstellation(std::vector<Point>* pSourceConstellation,
  std::vector<Point>* pSub0,
  std::vector<Point>* pSub1,
  std::vector<Point>* pSub2,
  std::vector<Point>* pSub3,
  double originX,
  double originY);

bool VerifyConstellationBounds(std::vector<Point>* pSourceConstellation, double maxX, double maxY, double minX, double minY);


unsigned int NaiveSearch(std::vector<  std::vector<Point>*  >* pTightCollection, PointDist &target, size_t maxPerQuad, Point *outPoints);

bool AddToQuad(PointDist &theNewPoint,
  PointDist* points,
  size_t &cursize,
  size_t &maxDistIndex,
  size_t &maxsize);

void DebugAddToQuad();

void FillInPointsArray(PointDist *pPointDistArr, size_t size1, Point *pPointArr, size_t start);


// CLASSES
class QTreeNode // can be a root node/leaf node, all-types
{
  /*
  Sub-Sector Diagram
  1  |  0
  |
  -----O-----
  |
  2  |  3
  */
public:
  QTreeNode* pSub0 = NULL; // 0
  QTreeNode* pSub1 = NULL; // 1
  QTreeNode* pSub2 = NULL; // 2
  QTreeNode* pSub3 = NULL; // 3

  QTreeNode* pParent = NULL; // used when not at level 1, for checking previous mid/extent points 

  int quadrant = -1; // -1, 0, 1, 2, 3 (denoting Sub-Sector Diagram) root has -1

  // max values for THIS node (encompassing all of the subnodes)
  double maxX = 0;
  double maxY = 0;
  double minX = 0;
  double minY = 0;
  // calculated after getting the max extents for THIS's constellation.
  double midpointX = 0.0;
  double midpointY = 0.0;

  unsigned int current_level;

  std::vector<Point>* pConstellation; // points this level holds (during construction) until its subdivided...

  QTreeNode(std::vector<Point>* pInConstellation, int level = 1, QTreeNode* pCreatedBy = NULL, int inQuadrant = -1)
  {
    // pInConstellation was caller-allocated. it has a subset of points from the previous call which "this" then further subdivide on, if necessary. 

    this->pConstellation = pInConstellation;

    this->current_level = level;

    this->pParent = pCreatedBy;

    this->quadrant = inQuadrant;

    if (this->current_level == 1) // level = 1 (root) should calculate, all other should just be derived
    {
      FindConstellationExtents(this->pConstellation, this->maxX, this->maxY, this->minX, this->minY);
    }
    else // you have a parent and you can see its midpoint/extents which will guide you in setting yours...
    {
      double oldMidX = this->pParent->midpointX;
      double oldMidY = this->pParent->midpointY;

      double oldMinX = this->pParent->minX;
      double oldMaxX = this->pParent->maxX;
      double oldMinY = this->pParent->minY;
      double oldMaxY = this->pParent->maxY;

      if (this->quadrant == 0)
      {
        this->minX = oldMidX;
        this->maxX = oldMaxX;
        this->minY = oldMidY;
        this->maxY = oldMaxY;
      }
      else
        if (this->quadrant == 1)
        {
        this->minX = oldMinX;
        this->maxX = oldMidX;
        this->minY = oldMidY;
        this->maxY = oldMaxY;
        }
        else
          if (this->quadrant == 2)
          {
        this->minX = oldMinX;
        this->maxX = oldMidX;
        this->minY = oldMinY;
        this->maxY = oldMidY;
          }
          else
            if (this->quadrant == 3)
            {
        this->minX = oldMidX;
        this->maxX = oldMaxX;
        this->minY = oldMinY;
        this->maxY = oldMidY;
            }
            else
            {
              std::cout << "ERROR: this->quadrant not set correctly !!! (while creating QTreeNode()) " << std::endl;
              return;
            }

      /* VERFIY that the constellation, which was passed in from parent, lies entirely within our derived min/max boundaries...
      if (!VerifyConstellationBounds(this->pConstellation, this->maxX, this->maxY, this->minX, this->minY))
      {
        std::cout << "ERROR: VerifyConstellationBounds() FAILED (while creating QTreeNode()) " <<
          "LEVEL:" << this->current_level << " Quadrant:" << this->quadrant << std::endl;
        return;
      }*/
    }

    // DEEPER: if the size of the passed input constellation is <= QTREE_MAX_POINTS then we have subdivided enough,
    // keep the constellation here and dont subdivide further 
    // TODO: this condition assumes all areas of the constelllation are well-distributed, ie. some could fail to have enough
    // and not be worth subdividing on, hence you get a leaf higher-up. 5,000,000 pts and 7 levels yields a well-distributed cloud howevr ~1k per leaf
    if ((this->current_level < QTREE_MAX_LEVELS) && (this->pConstellation->size() > QTREE_MAX_POINTS)) // then we need to subdivide
    {
      // only need midpoints when you feel you need to go deeper
      this->midpointX = (minX + maxX) / 2;
      this->midpointY = (minY + maxY) / 2;

      /*
      Sub-Sector Diagram
      1  |  0
      |
      -----O-----
      |
      2  |  3
      */
      std::vector<Point>* pSub0Points = new std::vector<Point>();
      std::vector<Point>* pSub1Points = new std::vector<Point>();
      std::vector<Point>* pSub2Points = new std::vector<Point>();
      std::vector<Point>* pSub3Points = new std::vector<Point>();

      // populate each sub nodes' vector with the appropriate points....
      ChopConstellation(
        this->pConstellation,
        pSub0Points,
        pSub1Points,
        pSub2Points,
        pSub3Points,
        this->midpointX,
        this->midpointY);

      // Check to see that the sizes match up...
      if (this->pConstellation->size() == (pSub0Points->size() + pSub1Points->size() + pSub2Points->size() + pSub3Points->size()))
      {
        //std::cout << "BEFORE/AFTER SIZES checked out !!! " << std::endl;

        delete this->pConstellation;
        this->pConstellation = NULL; // at this point this node no longer has to hold the values, its children have them...

        // now create the four new sub nodes and assign them the points...
        this->pSub0 = new QTreeNode(pSub0Points, (this->current_level + 1), this, 0); // 0
        this->pSub1 = new QTreeNode(pSub1Points, (this->current_level + 1), this, 1); // 1
        this->pSub2 = new QTreeNode(pSub2Points, (this->current_level + 1), this, 2); // 2
        this->pSub3 = new QTreeNode(pSub3Points, (this->current_level + 1), this, 3); // 3
      }
      else
      {
        std::cout << "BEFORE/AFTER SIZES MISMATCH !!! (while creating QTreeNode()) " << std::endl;
        return;
      }
    }
  };

  ~QTreeNode()
  {

    if (this->pConstellation != NULL)
    {
      delete this->pConstellation;
      this->pConstellation = NULL;
    }

    if (this->pSub0 != NULL)
    {
      delete this->pSub0;
      this->pSub0 = NULL;
    }
    if (this->pSub1 != NULL)
    {
      delete this->pSub1;
      this->pSub1 = NULL;
    }
    if (this->pSub2 != NULL)
    {
      delete this->pSub2;
      this->pSub2 = NULL;
    }
    if (this->pSub3 != NULL)
    {
      delete this->pSub3;
      this->pSub3 = NULL;
    }

  };

  std::string ToString()
  {
    return "";
  };

  std::string WalkTree() // walk the graph and format it to a reversed string.
  {
    QTreeNode* pCur = this;
    std::string ret = "";

    while (true)
    {
      if (ret.size() > 0) // if already something there
        ret = "-" + ret;

      ret = "(" + std::to_string(pCur->current_level) + ":" + std::to_string(pCur->quadrant) + ")" + ret;

      if (pCur->pParent != NULL)
        pCur = pCur->pParent;
      else
        break;
    }
    return ret;
  };

  /* WalkTree() test
  std::cout << "-> " << pQ->WalkTree() << std::endl;
  std::cout << "-> " << pQ->pSub0->WalkTree() << std::endl;
  std::cout << "-> " << pQ->pSub0->pSub0->WalkTree() << std::endl;
  std::cout << "-> " << pQ->pSub0->pSub0->pSub3->pSub2->pSub0->pSub3->WalkTree() << std::endl;
  */

};



bool ConstellationBoundsPointDist(QTreeNode* pNode, PointDist &pd);

QTreeNode* SettlePointDistIntoQTree(QTreeNode* currentNode, PointDist &pd);

void GetPointsUnderNode(QTreeNode* pNode, std::vector<std::vector<Point>*>* arrayOfArrays); // used to be just an fn

size_t SmahtSearch(QTreeNode* pNode, PointDist &target, size_t maxPerQuad, Point* outPointsArray, bool bPrintDebug = true);


// Class-Dependant Structs
struct Object // "handle" the original caller application used for the trasaction.
{
  size_t globalcounter1 = 0; // counts querries

  QTreeNode* pRootNode = NULL;

  size_t totalnumberpoints = 0;

  ~Object()
  {
    //printf("OBJECT Destructor Called.\n");
    if (this->pRootNode != NULL)
    {
      delete this->pRootNode;
      this->pRootNode = NULL;
    }
  };
};

void SearchWrapper(Object* pObj);

void RunWrapper(int iterations);

bool Approaches(double a, double b, double tolerance);



// =============================================================================================
// API Stuff =========================================================================
// =============================================================================================
#define EXTERN_DLL_EXPORT extern "C" __declspec(dllexport) 
// API ENTRYPOINTS
EXTERN_DLL_EXPORT Object * Construct(const Point *points, size_t numPoints);
EXTERN_DLL_EXPORT size_t Search(Object *object, const Point& target, double searchRadius, size_t maxPerQuad, Point *outPoints);
EXTERN_DLL_EXPORT void Destroy(Object *object);