<?php

/*
*     License:  This  program  is  free  software; you can redistribute it and/or
*     modify it under the terms of the GNU General Public License as published by
*     the  Free Software Foundation; either version 3 of the License, or (at your
*     option)  any later version. This program is distributed in the hope that it
*     will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
*     Public License for more details.
*/

  // JOFLA.NET (2023)
  // Cert Checker
  // terminal application to "pin" https SSL certs / ssh keys and check them against whats currently being returned 
  // can be used to inventory a bunch of machines or just check for any funny business going on

  // the certchecker.json file needs to exist, 
  // with entries, at least empty ones, which can be populated with values, at runtime, otherwise program does nothing.
  // check <exampleHost> entries in provided json.
  // after successfully connecting for the first time the json will have hash entries for the fingerprints it encountered.
  // https entries will be stored under "SHA256" / SSH entries will have a "fingerprints" sub-array 
  $jsonFileStr = 'certchecker.json';

  function tprint($inStr)
  {
    echo($inStr . "\n");
  }

  function GetStorageArray($keyStr, $sha256Str)
  {
    return Array($keyStr => Array('SHA256' => $sha256Str));
  }

  function GetSSHCertScript()
  {
    return <<< EOT
#!/bin/bash

# SSH fingerprint (cert) checker

if [ -z "$1" ] # check if the first arg passed is empty
  then
    #echo "No Target Host Specified, Exiting.";
    exit;
  #else
    #echo "Host set to: $1";
fi

if [ ! -z "$2" ] # if a second arg specified, assume port
  then
    #echo "Port set to: $2";
    portsection=" -p $2 ";
  else
    portsection="";
fi

  firstcmd="ssh-keyscan \${portsection} \${1}"

  file=$(mktemp)

  \$firstcmd > \$file 

  echo "1st ReturnCode: $?"

  ssh-keygen -l -f \$file

  echo "2nd ReturnCode: $?"

  rm \$file

EOT;

  }

  function FetchSSHCertSigs($addressPortStr) // ie. "127.0.0.1 22"
  {
    $addressPortStr = str_replace(':', ' ', $addressPortStr);

    $targetscript = 'temp-ssh-cert-check.sh';

    if( file_put_contents($targetscript, GetSSHCertScript()) !== false)
    {
      if(chmod($targetscript, 0755))
      {
        $ret = shell_exec("./" . $targetscript . " $addressPortStr " . " 2>&1"); // stderr to stdout 

        unlink($targetscript);
        
        // Parse the return
        // In between:
        // 1st ReturnCode: 0
        //  <signatures to verify/store>
        // 2nd ReturnCode: 0

        $retArr = explode("\n", $ret);

        //echo(" " . print_r($retArr, true) . "\n");

        $firstZero = -1;
        $secondZero = -1;
        $retArrLen = count($retArr);
        for($i = 0; $i < $retArrLen; $i++){
          if($retArr[$i] == "1st ReturnCode: 0")
            $firstZero = $i;
          else if($retArr[$i] == "2nd ReturnCode: 0")
            $secondZero = $i;
        }

        if(($firstZero != -1) && ($secondZero != -1) && ($firstZero < $secondZero)) // valid row states
        {

          $sshFingerprintsArr = Array( );

          for($i = $firstZero + 1; $i < $secondZero; $i++)
          {

            $line = trim($retArr[$i]);

            
            $pos1 = strrpos($line, '(');
            $pos2 = strrpos($line, ')');

            if( $pos1 !== false && $pos2 !== false)
            {
              $method = substr($line , $pos1 + 1, $pos2 - $pos1 - 1); // method without parens

              $lineArr = explode(' ', $line);

              $sshFingerprintsArr[$method . '-' . $lineArr[0]] =  $lineArr[1] ;
            }                     

          }

          return $sshFingerprintsArr;
        }
        else // error occurred somewhere
        {
          return "ERROR while parsing return of '$targetscript' for $addressPortStr\n";
          // could be many things, ie. name resolution, incorrect address/port, and or protocol
        }
      }
      else
      {
        unlink($targetscript);
        die("Fatal Error: Could not chmod $targetscript script!\n");
      }
    }
    else
    {
      die("Fatal Error: Could not write temporary $targetscript script!\n");
    }

    return "Error!\n";
  }

  function FetchEvalCertSHA256($addressPortStr)
  {
    $ret = shell_exec("timeout 15s openssl s_client -connect ".$addressPortStr." < /dev/null 2>/dev/null | openssl x509 -fingerprint -sha256 -noout -in /dev/stdin 2>/dev/null");

    $validReturnPrefixStr = 'SHA256 Fingerprint='; // valid outputs start with this 

    if(($ret !== false) && ($ret !== null) && is_string($ret) &&  
      (strlen($ret) >= strlen($validReturnPrefixStr)) &&
      substr($ret,0,strlen($validReturnPrefixStr)) == $validReturnPrefixStr )
    {
      $rest = trim(substr($ret, strlen($validReturnPrefixStr))); // should be the hash...
      return $rest;
    }
    else
      return false; // error occurred, ie no server at address or something
  }

  function CommandExists($cmdStr) 
  {
    $returnVal = shell_exec(sprintf("which %s", escapeshellarg($cmdStr)));
    return !empty($returnVal);
  }

  // ================      ================================================================
  // ================ MAIN ================================================================
  // ================      ================================================================

  if(PHP_OS != "Linux" || php_sapi_name() != 'cli')
    die("Script Was designed to be run in BASH Linux only.");

  $dependanciesArr = Array(
    'openssl',
    'ssh-keyscan',
    'ssh-keygen'
  );
  foreach($dependanciesArr as $entry)
    if(!CommandExists($entry))
      die("Error: Dependancy command '$entry' was not found on this system. Please Install.\n");

  tprint("Cert Checker v0.7\n");

  $bChangesMade = false;

  $jsonOBJArr = NULL;

  if(is_file($jsonFileStr))
    $jsonOBJArr = json_decode(file_get_contents($jsonFileStr), true);

  if(is_array($jsonOBJArr))
  {
    // TOP LEVEL JSON OBJECT LOOP
    foreach($jsonOBJArr as $key => $value) // keys are <ip>:<port>, vals are Array('<HASH-TYPE>' => '<ACTUAL-HASH>')
    {
      //TODO: add key validators for <ip>:<port>

      // Skip over example entries 
      $examplePrefix = '<exampleHost>';
      if( (mb_strlen($key) >= mb_strlen($examplePrefix)) && 
        ( mb_substr($key, 0, mb_strlen($examplePrefix)) == $examplePrefix) )
        continue;

      
      if(is_array($value))
      {
        if(isset($value['proto']) && is_string($value['proto']))
        {
          if($value['proto'] == 'ssh')
          {
            // Comment
            $commentStr = '';
            if(is_array($value) && isset($value['comment']) && is_string($value['comment']))
              $commentStr = $value['comment'];

            if(strlen($commentStr) > 0)
              tprint('(SSH) ' . $commentStr.':');
            else
              tprint('(SSH) ' . $key); // else just use the key
            
            // Make the query
            $sshFetchedArr = FetchSSHCertSigs($key);

            if(is_array($sshFetchedArr)) // success
            {
              // if we already have fingerpints set, compare them to whats just come back.
              if(isset($value['fingerprints']) && is_array($value['fingerprints']))
              {
                $storedFingerprintArr = $value['fingerprints'];

                $bAllMatched = true;
                // compare each existing type of subprint to what was just querried
                // warn if one is missing/wrong. error if none match!
                foreach($storedFingerprintArr as $keyMethod => $valueHash)
                {
                  // if match
                  if(isset($sshFetchedArr[$keyMethod]) && 
                    strcmp($sshFetchedArr[$keyMethod], $valueHash) === 0
                  )
                  {

                  }
                  else // if there is/are a mismatch, itemize it.
                  {
                    tprint('SSH Hash Mismatch❗');
                    tprint("Auth Method: " . $keyMethod);
                    tprint('Existing Hash:' . $valueHash);
                    tprint('Current  Hash:' . $sshFetchedArr[$keyMethod]);

                    $bAllMatched = false;
                  }
                }

                if($bAllMatched) // all matched
                {
                  
                  tprint("OK: ALL SSH fingerprint(s) from \"" . $key . "\" MATCH ✅\n Stored Fingerprint(s) \n " . print_r($storedFingerprintArr, true) . "");
                }

              }
              else // was empty so just set it
              {
                $jsonOBJArr[$key]['fingerprints'] = $sshFetchedArr;

                $jsonOBJArr[$key]['setdate'] = date(DATE_RFC2822);
                
                $bChangesMade = true;

                tprint("Updated this entry, recheck .json file.");
              }
            }
            else // failed, print the string
            {
              tprint( $sshFetchedArr );
            }
          }
          else
          if($value['proto'] == 'https')
          {
            // Comment if present
            $commentStr = '';
            if(is_array($value) && isset($value['comment']) && is_string($value['comment']))
              $commentStr = $value['comment'];

            if(strlen($commentStr) > 0)
              tprint('(HTTPS) ' . $commentStr.':');
            else
              tprint('(HTTPS) ' . $key); // else just use the key


            $storedFingerprintStr = '';

            if(isset($value['SHA256']) && is_string($value['SHA256']))
              $storedFingerprintStr = $value['SHA256'];


            $currentFingerprintStr = FetchEvalCertSHA256($key);

            if($currentFingerprintStr !== false) // querried something successful
            {
              if( strlen($storedFingerprintStr) > 0 ) // stored was filled-in
              {
                if(strcmp($storedFingerprintStr, $currentFingerprintStr) !== 0 )
                {
                  tprint("<<<WARNING>>>: Certificate fingerprint from \"" . $key . "\" DIFFERENT ❌\n Stored Fingerprint: \n  SHA256 (".$storedFingerprintStr.") \n New Returned Fingerprint: \n  SHA256 (".$currentFingerprintStr.")");
                }
                else // SAME
                {
                  tprint("OK: Certificate fingerprint from \"" . $key . "\" MATCHES ✅\n Stored Fingerprint \n  SHA256 (" . $storedFingerprintStr . ")");
                }
              }
              else // if stored was empty we'll write-out whatever comes back this time from executing FetchEvalCertSHA256()
              {
                $jsonOBJArr[$key]['SHA256'] = $currentFingerprintStr;

                $jsonOBJArr[$key]['setdate'] = date(DATE_RFC2822);
                
                $bChangesMade = true;

                tprint("Updated this entry, recheck .json file.");
              }
            }
            else
              tprint("ERROR: FAILED TO RETRIEVE SSL/TLS CERTIFICATE from \"" . $key . "\" ❗");
          }
          else
            continue; // TODO: some kind of error message 
        }      
      }

      tprint('');
    }

    // Write out if changes made
    if($bChangesMade)
    {
      $encoded_file = json_encode( $jsonOBJArr , JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);

      $last_err = json_last_error();
      if($last_err === JSON_ERROR_NONE)
      {
        $tempName = $jsonFileStr . '.TEMP';
        if((file_put_contents($tempName, $encoded_file) !== false) && (filesize($tempName) > 0))
        {
          if(rename($tempName, $jsonFileStr) !== false)
          {
            tprint("One or more blank entries updated to current hash, check JSON file.");

            // it worked!
          }
          else
            tprint("ERROR: Failed to Rename Temp File to Target. Critical!");
        }
        else
          tprint("ERROR: Failed to Successfully write out Temp File. Aborting File Update.");
      }
      else
        tprint("ERROR: A JSON Encode Error ($last_err) Occurred. Aborting .json File Update.");

    }
  }
  else
    tprint("No JSON file present or file corrupt, terminating. (".$jsonFileStr.")");
