<?php

/*
*     License:  This  program  is  free  software; you can redistribute it and/or
*     modify it under the terms of the GNU General Public License as published by
*     the  Free Software Foundation; either version 3 of the License, or (at your
*     option)  any later version. This program is distributed in the hope that it
*     will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
*     Public License for more details.
*/

  // ====================================================================================================
  class FileLocker // Object conceptualization of the flock mechanism 
  {
    var $lockFileStr; // full/partial path to ".LOCK" file
    var $fp; // file pointer
    var $bLocked; // was successfully able to lock...
    
    function __construct($file)
    {
      if(is_dir($file) && (strcmp(mb_substr($file, -1), '/') !== 0) ) // folders can be referenced via '/blah' or '/blah/' so we can't just tack on the '.LOCK' in that situation 
        $file .= '/';
      
      $this->lockFileStr = $file . '.LOCK';
      
      if(!file_exists($this->lockFileStr))
        touch($this->lockFileStr);
      
      $this->fp = fopen($this->lockFileStr, "r+");
      
      $this->bLocked = false;		
    }
    
    function __destruct()
    {
      if($this->fp)
        fclose($this->fp); // lil cleanup 
    }
    
    function Lock()
    {
      if($this->bLocked === false) // intended to enforce correct usage of THIS object only 
      {
        if($this->fp && flock($this->fp, LOCK_EX)) // blocking call 
        {
          $this->bLocked = true;
          return true;
        } 
        else // wont happen, since our 'if' should block, unless windows, yetch 
        {
          error_log("FileLocker: Unable to lock file (".$this->lockFileStr.")");
          return false;
        }        
      }
      else
        return false;
    }
    
    function Release()
    {
      if($this->bLocked === true)
      {
        if($this->fp && flock($this->fp, LOCK_UN)) 
        {
          $this->bLocked = false;
          return true;
        }
        else
        {
          error_log("FileLocker: Unable to unlock file (".$this->lockFileStr.")");
          return false;          
        }
      }
      else
        return false;	
    }
  }
  // ====================================================================================================
  function FmtTimeIntoDate($time)
  {
    if(is_numeric($time))
      return date('Y/m/d h:i:s a', $time); 
    else
      return 'n/a';
  }
  // ====================================================================================================
  function StringifyUserAgts($inArr)
  {
    if(is_array($inArr))
    {
      $retStr = '';
      $bFirst = true;
      foreach($inArr as $key => $val)
      {
        if($bFirst) // only print <br>s on subsequent iteratiosn 
          $bFirst = false;
        else
          $retStr .= "<br>";
        
        if(is_string($key))
          $retStr .= "(" . htmlspecialchars($val) . ") <i>" . htmlspecialchars($key) . "</i>";
        else
          $retStr .= "<b>Not-A-String</b>";
      }
      return $retStr;
    }
    else
      return "Error(Not Array)";
  }
  // ====================================================================================================
  function PruneAddressArray($curTime, &$inArr, $lifetimeInDays) // O(n) Take the address array and remove "old" entries, return size of new arr
  {
    foreach($inArr as $key => $value)
    {
      $arr1 = explode('/', $value[1]);
      $latestTime = $arr1[1];
      if(($curTime - $latestTime) > ($lifetimeInDays * (60*60*24)))
      {
        unset($inArr[$key]);
      }
    }
    return count($inArr);
  }
  // ====================================================================================================
  function IsValidPositiveInteger($inStr)
  {
    if(
      is_string($inStr) && 
      (strcmp($inStr, strval(intval($inStr))) === 0) && 
      (intval($inStr) > 0)
    )
      return true;
    return false;
  }
  // ====================================================================================================
  function ReComitTheFile($filenameStr, $callerFilename, $addressArr, $entireArr) // recommit the update
  {
    if(is_array($entireArr)) // Merge the Updated key (callerFilename) into the larger collection
      $entireArr[$callerFilename] = $addressArr;
    else // or we're the only key so start from scratch
      $entireArr = Array($callerFilename => $addressArr);


    //json_encode requires UTF-8 encoded data. '\x80' (128) is the smallest sequence you can use to test it.
    $encoded_file = json_encode( $entireArr , JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);

    $last_err = json_last_error();
    if($last_err === JSON_ERROR_NONE)
    {
      $tempName = $filenameStr . '.TEMP';
      if((file_put_contents($tempName, $encoded_file) !== false) && (filesize($tempName) > 0))
      {
        if(rename($tempName, $filenameStr) !== false)
        {
          return true; // it worked!
        }
        else
          error_log("ERROR: (TrafficLogger) Failed to Rename Temp File to Target. Critical!");
      }
      else
        error_log("ERROR: (TrafficLogger) Failed to Successfully write out Temp File. Aborting File Update.");
    }
    else
      error_log("ERROR: (TrafficLogger) A Json Encode Error ($last_err) Occurred. Aborting File Update.");

    return false;
  }
  // ====================================================================================================
  function GetValidTimezonesList() // extend this list to support other regions
  {
    return Array
    (
      'America/New_York',
      'America/Chicago',
      'America/Denver',
      'America/Los_Angeles',
      'Europe/London',
      'Europe/Paris',
      'Europe/Istanbul'
    );
  }
  // ====================================================================================================
  function PrintViewFormMarkup($curTimezoneStr, $loaded, $possibilitiesArr)
  {
    $toRet = '';

    $arrOfTimeZones = GetValidTimezonesList();
    
    
    $toRet .= "<form action=\".\" method=\"GET\" accept-charset=\"UTF-8\">\r\n";
    $toRet .= "<input type=\"hidden\" name=\"view\" value=\"\">\r\n";
    
    $toRet .= "<table style=\"border-radius: 6px; background-color: #F0F0FF; padding: 4px;\">\r\n";
    
    // <tr>
    $toRet .= "<tr><td><b>Relative Timezone</b></td>\r\n";
    $toRet .= "<td><select name=\"tz\">\r\n";
    foreach($arrOfTimeZones as $zone)
    {
      if(strcmp($zone, $curTimezoneStr) === 0)
        $toRet .= "<option value=\"".$zone."\" selected=\"selected\">".$zone."</option>\r\n";
      else
        $toRet .= "<option value=\"".$zone."\">".$zone."</option>\r\n";
    }
    $toRet .= "</select></td>\r\n";
    $toRet .= "<td><i>Current: " . htmlspecialchars($curTimezoneStr) . "</i></td></tr>\r\n";
    
    // <tr>
    $toRet .= "<tr><td><b>Available Page</b></td>\r\n";
    $toRet .= "<td><select name=\"page\">\r\n";
    foreach($possibilitiesArr as $page)
    {
      if(strcmp($loaded, $page) === 0)
        $toRet .= "<option value=\"".htmlspecialchars($page)."\" selected=\"selected\">".htmlspecialchars($page)."</option>\r\n";
      else
        $toRet .= "<option value=\"".htmlspecialchars($page)."\">".htmlspecialchars($page)."</option>\r\n";
    }
    $toRet .= "</select></td>\r\n";
    $toRet .= "<td><i>Current Page: " . htmlspecialchars($loaded) . "</i></td></tr>\r\n";
    
    // <tr> Radio Buttons For Type of Sort 
    $toRet .= "<tr><td><b>Sort by : </b></td>
    <td colspan=\"2\">
    
      <input type=\"radio\" id=\"sortip\" name=\"sort\" value=\"ip\">
      <label for=\"sortip\">IP</label>
      
      <input type=\"radio\" id=\"sorthits\" name=\"sort\" value=\"hits\">
      <label for=\"sorthits\">Hits</label>
      
      <input type=\"radio\" id=\"sortearliest\" name=\"sort\" value=\"earliest\">
      <label for=\"sortearliest\">Earliest</label>
      
      <input type=\"radio\" id=\"sortlatest\" name=\"sort\" value=\"latest\">
      <label for=\"sortlatest\">Latest</label>
      
    </td></tr>\r\n";
    
    // Force Pruner
    $toRet .= "<tr><td><b>Force Prune : </b></td>
    <td colspan=\"2\"><input type=\"text\" id=\"forceprune\" name=\"".TRAFLGR_PRUNE_PARAM."\" size=\"3\"></input> <i>entries whose latest has been over this many days are removed</i></td>
    </tr>\r\n";
    
    // Button 
    $toRet .= "<tr><td colspan=3>\r\n";
    $toRet .= "<div align=right><input type=\"submit\" value=\"Refresh Page\"></div>\r\n";
    $toRet .= "</td></tr>\r\n";
    
    $toRet .= "</table>\r\n";
    $toRet .= "</form>\r\n";

    echo($toRet);
  }
  // ====================================================================================================
  function IPV4AddressKeyCompare($a, $b) // given 2 strings denoting ipv4 addresses
  {
    $aArr = explode('.', $a);
    $bArr = explode('.', $b);
    
    // ensure each part is correctly zero-prefixed
    foreach($aArr as &$elem)
    {
      if(strlen($elem) == 1)
        $elem = '00' . $elem;
      else 
      if(strlen($elem) == 2)
        $elem = '0' . $elem;
    }
    foreach($bArr as &$elem)
    {
      if(strlen($elem) == 1)
        $elem = '00' . $elem;
      else 
      if(strlen($elem) == 2)
        $elem = '0' . $elem;
    }
    $a = implode('', $aArr);
    $b = implode('', $bArr);
    return strcmp($a, $b);
  }
  // ====================================================================================================
  function HitsCompare($a, $b) // inputs are values of addressArr
  {
    if(is_array($a) && isset($a[0]) && is_array($b) && isset($b[0]))
    {
      $a = intval($a[0]);
      $b = intval($b[0]);
      
      if($a < $b)
        return -1;
      else if($b < $a)
        return 1;
      else 
        return 0;
    }
    else
      return 0;    
  }
  // ====================================================================================================
  function EarliestCompare($a, $b)
  {
    if(is_array($a) && isset($a[1]) && is_array($b) && isset($b[1]))
    {
      $aArr = explode('/',$a[1]);
      $bArr = explode('/',$b[1]);
      
      $a = intval($aArr[0]); // load earliest
      $b = intval($bArr[0]); // load earliest
      
      if($a < $b)
        return -1;
      else if($b < $a)
        return 1;
      else 
        return 0;
    }
    else
      return 0; 
  }
  // ====================================================================================================
  function LatestCompare($a, $b)
  {
    if(is_array($a) && isset($a[1]) && is_array($b) && isset($b[1]))
    {
      $aArr = explode('/',$a[1]);
      $bArr = explode('/',$b[1]);
      
      $a = intval($aArr[1]); // load latest
      $b = intval($bArr[1]); // load latest
      
      if($a < $b)
        return -1;
      else if($b < $a)
        return 1;
      else 
        return 0;
    }
    else
      return 0; 
  }
  // ====================================================================================================
  // ====================================================================================================
  // ====================================================================================================
  // ====================================================================================================
  
  // Config Constants (strings) thees 'need' to be declared globally so i smurf-named it 
  const TRAFLGR_FILE_STORE = 'TrafficLoggerStore.json'; // where everything gets stored for this instance...
  const TRAFLGR_VIEW_PARAM = '<VIEWPARAMETER>'; // the param passed to view the current state; kind of like a password
  const TRAFLGR_ENTRY_LIFETIME = '60'; // (in days) when we load from the file store, if more than this amount has passed, the entry is considered stale
  const TRAFLGR_PRUNE_PARAM = 'prune'; // if this and VIEW_PARAM set, if entry is over this thresh then prune it (in days)
  
  /*
  //Invocation Example
  <?php
    if ((include '../include/TrafficLogger.php') == TRUE)
      TrafficLoggerInvoke(__FILE__);
    else
      error_log('Error: TrafficLogger.php could not be included.');
  */
  
  function TrafficLoggerInvoke($callerFilename) // Public call-in point
  {
    // output file will be a relative path, from the location of THIS script file.
    $filenameStr = dirname(__FILE__) . DIRECTORY_SEPARATOR . TRAFLGR_FILE_STORE; 
    
    $jsonArr = false; // uninitialized state; remains so if the filenameStr was empty nonexistant or corrupt
    
    $callerStringLoaded = 'NA';
    
    $size_addressArr = 0;
    
    $addressArr = Array(); // array, listing of whose visited this page, thusly 
    /*
      [IP] => [ COUNT / FIRST/LATEST / HostLookup / Array(<useragents>)]
    */
    
    $curCliIPStr = 'NA';
    if(isset($_SERVER['REMOTE_ADDR']))
      $curCliIPStr = $_SERVER['REMOTE_ADDR'];
    
    $curUserAgentStr = 'NA';
    if(isset($_SERVER['HTTP_USER_AGENT']))
    {
      $curUserAgentStr = $_SERVER['HTTP_USER_AGENT'];
      
      // curl "http://<target>" -A $'\x80' > /dev/null
      if(mb_check_encoding($curUserAgentStr, 'UTF-8') === false)
      {
        $curUserAgentStr = 'BASE64('. base64_encode($curUserAgentStr) .')';
        error_log("ERROR: (TrafficLogger) HTTP user agent not in UTF-8 : " . $curUserAgentStr); 
      }
    }
    
    // LOCK
    $lockObj = new FileLocker($filenameStr);
    $lockObj->Lock();
    
    $curTime = time(); // cur time used in various places for entire execution

    // LOADER
    if(file_exists($filenameStr) && filesize($filenameStr) > 0)
      if(($fileStr = file_get_contents($filenameStr)) !== false ) // if the file converted to a string is valid 
        if(($jsonArr = json_decode($fileStr, true)) !== NULL) // if we have valid json to play with 
          if(is_array($jsonArr))
          {
            // Cleanup old arr format which used 0,1 keyings
            if(isset($jsonArr[0])) unset($jsonArr[0]);
            if(isset($jsonArr[1])) unset($jsonArr[1]);
            
            // should only be able to adjust the 'viewable' page if we're viewing and such a page already exists in the database 
            if(isset($_GET[TRAFLGR_VIEW_PARAM]) && isset($_GET['page']) && is_string($_GET['page']))
            {
              if(isset($jsonArr[$_GET['page']]))
                $callerFilename = $_GET['page'];
              else
                error_log("Testing: BOGUS KEY FOUND ON VIEWING");
            }
            
            if(isset($jsonArr[$callerFilename]) && is_array($jsonArr[$callerFilename]))
            {
              // jsonArr should be "callerFilename" => "addressArr", it may have several pages' keys and address arrays 
              $addressArr = $jsonArr[$callerFilename];
              $size_addressArr = count($addressArr);
              $callerStringLoaded = $callerFilename;
            }
          }

    // at this point we either have something from the file or an empty array, but an array (addressArr) nonetheless
    
    $size_old = $size_addressArr; // store the old size
    
    // PRUNING
    if
    (isset($_GET[TRAFLGR_VIEW_PARAM]) && isset($_GET[TRAFLGR_PRUNE_PARAM]) && is_string($_GET[TRAFLGR_PRUNE_PARAM]) && IsValidPositiveInteger($_GET[TRAFLGR_PRUNE_PARAM]))
      $size_addressArr = PruneAddressArray($curTime, $addressArr, $_GET[TRAFLGR_PRUNE_PARAM]);
    else
    if(rand(1,20) == 1) // non-forced, randomly invoked, cleanup %05
      $size_addressArr = PruneAddressArray($curTime, $addressArr, TRAFLGR_ENTRY_LIFETIME);


    if(isset($_GET[TRAFLGR_VIEW_PARAM]) && $size_addressArr > 0) // view whats stored
    {
      $bSortOccurred = false;
      
      // Timezone
      $initTimezoneStr = date_default_timezone_get();
      $curTimezoneStr = '';
      if(isset($_GET['tz']) && is_string($_GET['tz']) && in_array($_GET['tz'], GetValidTimezonesList()))
      {
        $curTimezoneStr = $_GET['tz'];
        if(!date_default_timezone_set($curTimezoneStr))
          $curTimezoneStr = $initTimezoneStr; // if failure, revert 
      }
      else // default tz behavior
        $curTimezoneStr = $initTimezoneStr; // at this point 'cur' could be something not in GetValidTimezonesList()


      // Sorting (allow sort on 'viewing' only)
      // this rewrites the keys in the addressArr such that when we view without sorting we see the last sort + any new hits tacked on the end.
      if(isset($_GET['sort']) && is_string($_GET['sort']))
      {
        if(strcmp($_GET['sort'], 'ip') === 0)
          uksort($addressArr, 'IPV4AddressKeyCompare');
        else if(strcmp($_GET['sort'], 'hits') === 0)
          uasort($addressArr, 'HitsCompare');
        else if(strcmp($_GET['sort'], 'earliest') === 0)
          uasort($addressArr, 'EarliestCompare');
        else if(strcmp($_GET['sort'], 'latest') === 0)
          uasort($addressArr, 'LatestCompare');
        
        $bSortOccurred = true;
      }

      // View Page Markup
      echo("<!DOCTYPE html>\r\n");
      echo("<head><meta charset=\"utf-8\"><title>TrafficLogger (".htmlspecialchars($callerStringLoaded).")</title></head>\r\n<body>\r\n");
      PrintViewFormMarkup($curTimezoneStr, $callerStringLoaded, array_keys($jsonArr));
      echo("<p>Page: <i>" . htmlspecialchars($callerStringLoaded) . "</i> Expiry: Latest + ".TRAFLGR_ENTRY_LIFETIME." Day(s). Unique IP Entries:".$size_addressArr."</p>\r\n");
      echo("<table cellpadding=4>\r\n");
      echo("<tr style=\"background-color:#F0F0FF\"><td><b>IP/Hostname</b></td><td><b>Hits</b></td><td><b>Earliest/Latest</b></td><td><b>Useragent(s)</b></td></tr>\r\n");
      $bColor = true;
      foreach($addressArr as $key => $valArr)
      {
        if($bColor) // oscillating background tr color 
        {
          $colorVals = 'FDFDFD'; // light
          $bColor = false;
        }
        else
        {
          $colorVals = 'F3F3F3'; // dark
          $bColor = true;
        }
        $timeAsArr = explode('/',$valArr[1]);
        $startTime = FmtTimeIntoDate($timeAsArr[0]);
        $latestTime = FmtTimeIntoDate($timeAsArr[1]);
        echo("
<tr style=\"background-color:#".$colorVals."\">
  <td>".htmlspecialchars($key)."<br><i>".htmlspecialchars($valArr[2])."</i></td>
  <td>".htmlspecialchars($valArr[0])."</td>
  <td>".$startTime."<br>".$latestTime."</td>
  <td>".StringifyUserAgts($valArr[3])."</td>
</tr>\r\n
");
      }
      echo("</table>\r\n");
      echo("</body>\r\n</html>\r\n");
      
      if(($size_old != $size_addressArr) || $bSortOccurred) // recommit the file IFF the size changed due to pruning
        ReComitTheFile($filenameStr, $callerFilename, $addressArr, $jsonArr); // always recomit as this path always incurs change. 
      
      $lockObj->Release();
      die(); // no further execution should occur on a "view" 
    }
    else // log the current event
    {
      if(isset($addressArr[$curCliIPStr])) // if this key is 'set', then its been seen
      {
        $valArr = &$addressArr[$curCliIPStr]; // make new variable for shorthand but reference it since we want to edit whats in addressArr ultimately 
        
        $valArr[0]++; // increment counter
        
        // update time
        if(isset($valArr[1]) && is_string($valArr[1]))
          $valArr[1] = (explode('/',$valArr[1])[0]) .'/'. $curTime; // restamp the latest time entry (stored as raw seconds)
        
        // update the user-agent section (if we havent seen it tack on another, otherwise increment counter in the sub-array)
        if(isset($valArr[3]) && is_array($valArr[3]))
        {
          if(isset($valArr[3][$curUserAgentStr])) // if weve visited here via this useragent already
            $valArr[3][$curUserAgentStr] = $valArr[3][$curUserAgentStr] + 1;
          else
            $valArr[3][$curUserAgentStr] = 1;
        }
      }
      else // wasnt (set) found ; so set it with the following val array
        $addressArr[$curCliIPStr] = Array( 1, /*first/latest*/ $curTime.'/'.$curTime, gethostbyaddr($curCliIPStr), Array($curUserAgentStr => 1));
      
      ReComitTheFile($filenameStr, $callerFilename, $addressArr, $jsonArr); // always recomit as this path always incurs change. 
      
      $lockObj->Release();
    }

  }

